// Includes
//=========

#include "Graphics.h"

#include "cConstantBuffer.h"
#include "ConstantBufferFormats.h"
#include "cRenderState.h"
#include "cShader.h"
#include "cVertexFormat.h"
#include "sContext.h"
#include "VertexFormats.h"

#include <Engine/Asserts/Asserts.h>
#include <Engine/Concurrency/cEvent.h>
#include <Engine/Logging/Logging.h>
#include <Engine/Platform/Platform.h>
#include <Engine/ScopeGuard/cScopeGuard.h>
#include <Engine/Time/Time.h>
#include <Engine/UserOutput/UserOutput.h>
#include <new>
#include <utility>
#include <Engine/Graphics/cView.h>

// Static Data
//============

namespace
{
	// In Direct3D "views" are objects that allow a texture to be used a particular way:
	eae6320::Graphics::cView* s_View = nullptr;

	// Constant buffer object
	eae6320::Graphics::cConstantBuffer s_constantBuffer_frame(eae6320::Graphics::ConstantBufferTypes::Frame);
	eae6320::Graphics::cConstantBuffer s_constantBuffer_drawCall(eae6320::Graphics::ConstantBufferTypes::DrawCall);

	// Submission Data
	//----------------

	// This struct's data is populated at submission time;
	// it must cache whatever is necessary in order to render a frame
	struct sDataRequiredToRenderAFrame
	{
		eae6320::Graphics::ConstantBufferFormats::sFrame constantData_frame;
		float m_backgroundColor[3];
		uint16_t m_gameObjectCount;
		eae6320::Graphics::cMesh** m_meshArr;
		eae6320::Graphics::cEffect** m_effectArr;
		eae6320::Math::cMatrix_transformation* m_localToWorldMatrixArr;

		void CleanUp() {
			m_gameObjectCount = 0;
			delete[] m_meshArr;
			delete[] m_effectArr;
			delete[] m_localToWorldMatrixArr;
		}
	};

	// Skylar:
	// There are meshes that does not change frequently such as terrain, level layouts
	// These meshes does not need to be submitted every frame, so there is no need to Update() or so
	// This can improve performance, especially when there are many such terrain meshes 
	struct sDataStaticObject {
		uint16_t m_staticMeshCount = 0;
		eae6320::Graphics::cMesh** m_staticMeshArr;
		eae6320::Graphics::cEffect** m_staticEffectArr;

		void CleanUp() {
			m_staticMeshCount = 0;
			delete[] m_staticMeshArr;
			delete[] m_staticEffectArr;
		}
	};
	sDataStaticObject s_dataStaticObjects;

	// In our class there will be two copies of the data required to render a frame:
	//	* One of them will be in the process of being populated by the data currently being submitted by the application loop thread
	//	* One of them will be fully populated and in the process of being rendered from in the render thread
	// (In other words, one is being produced while the other is being consumed)
	sDataRequiredToRenderAFrame s_dataRequiredToRenderAFrame[2];
	auto* s_dataBeingSubmittedByApplicationThread = &s_dataRequiredToRenderAFrame[0];
	auto* s_dataBeingRenderedByRenderThread = &s_dataRequiredToRenderAFrame[1];
	// The following two events work together to make sure that
	// the main/render thread and the application loop thread can work in parallel but stay in sync:
	// This event is signaled by the application loop thread when it has finished submitting render data for a frame
	// (the main/render thread waits for the signal)
	eae6320::Concurrency::cEvent s_whenAllDataHasBeenSubmittedFromApplicationThread;
	// This event is signaled by the main/render thread when it has swapped render data pointers.
	// This means that the renderer is now working with all the submitted data it needs to render the next frame,
	// and the application loop thread can start submitting data for the following frame
	// (the application loop thread waits for the signal)
	eae6320::Concurrency::cEvent s_whenDataForANewFrameCanBeSubmittedFromApplicationThread;
}


// Interface
//==========

// Submission
//-----------

void eae6320::Graphics::SubmitElapsedTime(const float i_elapsedSecondCount_systemTime, const float i_elapsedSecondCount_simulationTime)
{
	EAE6320_ASSERT(s_dataBeingSubmittedByApplicationThread);
	auto& constantData_frame = s_dataBeingSubmittedByApplicationThread->constantData_frame;
	constantData_frame.g_elapsedSecondCount_systemTime = i_elapsedSecondCount_systemTime;
	constantData_frame.g_elapsedSecondCount_simulationTime = i_elapsedSecondCount_simulationTime;
}

void eae6320::Graphics::SubmitRenderData(sCamera* i_Camera, 
	float i_backgroundColor[], 
	uint16_t i_gameObjectCount, 
	cMesh** i_meshArr,
	cEffect** i_effectArr,
	Math::cMatrix_transformation* i_localToWorldMatrixArr)
{
	EAE6320_ASSERT(s_dataBeingSubmittedByApplicationThread);

	auto& backgroundColor = s_dataBeingSubmittedByApplicationThread->m_backgroundColor;
	for (int i = 0; i < 3; i++) {
		backgroundColor[i] = i_backgroundColor[i];
	}

	s_dataBeingSubmittedByApplicationThread->constantData_frame.g_transform_worldToCamera =
		Math::cMatrix_transformation::CreateWorldToCameraTransform(
			i_Camera->m_cameraOrientationPredicted,
			i_Camera->m_cameraPositionPredicted
		);

	s_dataBeingSubmittedByApplicationThread->constantData_frame.g_transform_cameraToProjected =
		Math::cMatrix_transformation::CreateCameraToProjectedTransform_perspective(
			i_Camera->m_verticalFieldOfView_inRadians,
			i_Camera->m_aspectRatio,
			i_Camera->m_z_nearPlane, i_Camera->m_z_farPlane
		);

	s_dataBeingSubmittedByApplicationThread->m_gameObjectCount = i_gameObjectCount;

	s_dataBeingSubmittedByApplicationThread->m_meshArr = i_meshArr;
	s_dataBeingSubmittedByApplicationThread->m_effectArr = i_effectArr;
	s_dataBeingSubmittedByApplicationThread->m_localToWorldMatrixArr = i_localToWorldMatrixArr;

}

void eae6320::Graphics::SubmitStaticObjectData(uint16_t i_staticMeshCount, 
	eae6320::Graphics::cMesh** i_staticMeshArr, eae6320::Graphics::cEffect** i_staticEffectArr)
{
	s_dataStaticObjects.m_staticMeshCount = i_staticMeshCount;
	s_dataStaticObjects.m_staticEffectArr = i_staticEffectArr;
	s_dataStaticObjects.m_staticMeshArr = i_staticMeshArr;
}



eae6320::cResult eae6320::Graphics::WaitUntilDataForANewFrameCanBeSubmitted(const unsigned int i_timeToWait_inMilliseconds)
{
	return Concurrency::WaitForEvent(s_whenDataForANewFrameCanBeSubmittedFromApplicationThread, i_timeToWait_inMilliseconds);
}

eae6320::cResult eae6320::Graphics::SignalThatAllDataForAFrameHasBeenSubmitted()
{
	return s_whenAllDataHasBeenSubmittedFromApplicationThread.Signal();
}

// Render
//-------

void eae6320::Graphics::RenderFrame()
{
	// Wait for the application loop to submit data to be rendered
	{
		if (Concurrency::WaitForEvent(s_whenAllDataHasBeenSubmittedFromApplicationThread))
		{
			// Switch the render data pointers so that
			// the data that the application just submitted becomes the data that will now be rendered
			std::swap(s_dataBeingSubmittedByApplicationThread, s_dataBeingRenderedByRenderThread);
			// Once the pointers have been swapped the application loop can submit new data
			if (!s_whenDataForANewFrameCanBeSubmittedFromApplicationThread.Signal())
			{
				EAE6320_ASSERTF(false, "Couldn't signal that new graphics data can be submitted");
				Logging::OutputError("Failed to signal that new render data can be submitted");
				UserOutput::Print("The renderer failed to signal to the application that new graphics data can be submitted."
					" The application is probably in a bad state and should be exited");
				return;
			}
		}
		else
		{
			EAE6320_ASSERTF(false, "Waiting for the graphics data to be submitted failed");
			Logging::OutputError("Waiting for the application loop to submit data to be rendered failed");
			UserOutput::Print("The renderer failed to wait for the application to submit data to be rendered."
				" The application is probably in a bad state and should be exited");
			return;
		}
	}



	EAE6320_ASSERT(s_dataBeingRenderedByRenderThread);

	// clear view
	if (s_View) {
		s_View->ClearView(s_dataBeingRenderedByRenderThread->m_backgroundColor[0],
			s_dataBeingRenderedByRenderThread->m_backgroundColor[1], 
			s_dataBeingRenderedByRenderThread->m_backgroundColor[2]);
	}

#if defined( EAE6320_PLATFORM_D3D )
	auto* const dataRequiredToRenderFrame = s_dataBeingRenderedByRenderThread;
#endif

	// Update the frame constant buffer
	{
		// Copy the data from the system memory that the application owns to GPU memory
#if defined( EAE6320_PLATFORM_D3D )
		auto& constantData_frame = dataRequiredToRenderFrame->constantData_frame;
#elif defined( EAE6320_PLATFORM_GL )
		auto& constantData_frame = s_dataBeingRenderedByRenderThread->constantData_frame;
#endif
		s_constantBuffer_frame.Update(&constantData_frame);
	}

	// draw mesh
	{
		eae6320::Graphics::ConstantBufferFormats::sDrawCall constantData_drawCall;
		// static meshes
		{
			s_constantBuffer_drawCall.Update(&constantData_drawCall);
			for (int i = 0; i < s_dataStaticObjects.m_staticMeshCount; i++) {
				s_dataStaticObjects.m_staticEffectArr[i]->BindEffect();
				s_dataStaticObjects.m_staticMeshArr[i]->DrawMesh();
			}
		}
		// dynamic meshes
		{
			uint16_t i_meshCount = s_dataBeingRenderedByRenderThread->m_gameObjectCount;
			for (int i = 0; i < i_meshCount; i++) {
				constantData_drawCall.g_transform_localToWorld =
					s_dataBeingRenderedByRenderThread->m_localToWorldMatrixArr[i];
				s_constantBuffer_drawCall.Update(&constantData_drawCall);

				s_dataBeingRenderedByRenderThread->m_effectArr[i]->BindEffect();
				s_dataBeingRenderedByRenderThread->m_meshArr[i]->DrawMesh();
			}
		}
		
	}

	// Everything has been drawn to the "back buffer", which is just an image in memory.
	// In order to display it the contents of the back buffer must be "presented"
	// (or "swapped" with the "front buffer", which is the image that is actually being displayed)
	{
		if (s_View) {
			s_View->SwapBackBuffer();
		}
	}


	// After all of the data that was submitted for this frame has been used
	// you must make sure that it is all cleaned up and cleared out
	// so that the struct can be re-used (i.e. so that data for a new frame can be submitted to it)
	{
		// (At this point in the class there isn't anything that needs to be cleaned up)
		//dataRequiredToRenderFrame	// TODO
		s_dataBeingRenderedByRenderThread->CleanUp();
	}
}

// Initialize / Clean Up
//----------------------

eae6320::cResult eae6320::Graphics::Initialize(const sInitializationParameters& i_initializationParameters)
{
	auto result = Results::Success;

	// Initialize the platform-specific context
	if (!(result = sContext::g_context.Initialize(i_initializationParameters)))
	{
		EAE6320_ASSERTF(false, "Can't initialize Graphics without context");
		return result;
	}
	// Initialize the platform-independent graphics objects
	{
		if (result = s_constantBuffer_frame.Initialize())
		{
			// There is only a single frame constant buffer that is reused
			// and so it can be bound at initialization time and never unbound
			s_constantBuffer_frame.Bind(
				// In our class both vertex and fragment shaders use per-frame constant data
				static_cast<uint_fast8_t>(eShaderType::Vertex) | static_cast<uint_fast8_t>(eShaderType::Fragment));
		}
		else
		{
			EAE6320_ASSERTF(false, "Can't initialize Graphics without frame constant buffer");
			return result;
		}
		if (result = s_constantBuffer_drawCall.Initialize())
		{
			// There is only a single frame constant buffer that is reused
			// and so it can be bound at initialization time and never unbound
			s_constantBuffer_drawCall.Bind(
				// In our class only vertex shaders use per-frame constant data
				static_cast<uint_fast8_t>(eShaderType::Vertex));
		}
		else
		{
			EAE6320_ASSERTF(false, "Can't initialize Graphics without draw call constant buffer");
			return result;
		}
	}
	// Initialize the events
	{
		if (!(result = s_whenAllDataHasBeenSubmittedFromApplicationThread.Initialize(Concurrency::EventType::ResetAutomaticallyAfterBeingSignaled)))
		{
			EAE6320_ASSERTF(false, "Can't initialize Graphics without event for when data has been submitted from the application thread");
			return result;
		}
		if (!(result = s_whenDataForANewFrameCanBeSubmittedFromApplicationThread.Initialize(Concurrency::EventType::ResetAutomaticallyAfterBeingSignaled,
			Concurrency::EventState::Signaled)))
		{
			EAE6320_ASSERTF(false, "Can't initialize Graphics without event for when data can be submitted from the application thread");
			return result;
		}
	}
	// Initialize the views
	{
		s_View = new cView();
		if (!(result = s_View->InitializeViews(i_initializationParameters)))
		{
			EAE6320_ASSERTF(false, "Can't initialize Graphics without the views");
			return result;
		}
	}

	return result;
}

eae6320::cResult eae6320::Graphics::CleanUp()
{
	auto result = Results::Success;

	// view clean up
	{
		if (s_View) {
			result = s_View->CleanUp();
			delete s_View;
		}
	}

	{
		const auto result_constantBuffer_frame = s_constantBuffer_frame.CleanUp();
		if (!result_constantBuffer_frame)
		{
			EAE6320_ASSERT(false);
			if (result)
			{
				result = result_constantBuffer_frame;
			}
		}
		
		const auto result_constantBuffer_drawCall = s_constantBuffer_drawCall.CleanUp();
		if (!result_constantBuffer_drawCall)
		{
			EAE6320_ASSERT(false);
			if (result)
			{
				result = result_constantBuffer_drawCall;
			}
		}
	}

	{
		const auto result_context = sContext::g_context.CleanUp();
		if (!result_context)
		{
			EAE6320_ASSERT(false);
			if (result)
			{
				result = result_context;
			}
		}
	}

	return result;
}

